############################################################################
###             Script for morphological data analyses                   ###
############################################################################

library(geomorph)
library(shapes)
library(ade4)

#####################################################################
#####################################################################
# 1. Extract morphological data
#####################################################################
#####################################################################

load("morphometric_data.R")

#####################################################################
## a. perform PCA on the hybrid population
#####################################################################

### i. extract only individuals form the hybrid population and make procrustes superimposition
hybrids<-tot[,,163:841]
semi.landmarks<-as.matrix(read.table(file.choose(),h=T))# choose the file "semi_landmarks.txt"
PROC<-gpagen(hybrids,curves=semi.landmarks)

### ii. calculate mean shape per individual
mean.shape.hybrids<-array(NA,dim=c(32,2,length(unique(dimnames(hybrids)[[3]]))))
for (point in (1:32)){
  for (xy in (1:2)){
    res<-tapply(PROC$coord[point,xy,],dimnames(hybrids)[[3]],mean)
    mean.shape.hybrids[point,xy,]<-res
  }
}
dimnames(mean.shape.hybrids)<-list(NULL,NULL,names(res))

### iii. Perform the PCA and extract pc coordinates for each individual
#create a matrix
PROC.t<-aperm(mean.shape.hybrids,c(3,1,2))
dim(PROC.t)
PROC.mat<-cbind(PROC.t[,,1],PROC.t[,,2])
rownames(PROC.mat)<-dimnames(mean.shape.hybrids)[[3]]
#make pca
pca<-dudi.pca(PROC.mat,scannf=T,scale=F)
pourcent<-(pca$eig/sum(pca$eig))*100
#extract pc coordinates for each individual
coord_hybpca<-cbind(rownames(pca$li),pca$li)

#####################################################################
## b. perform PCA on the parents
#####################################################################

# extract parents data
PROC.parents<-PROC.mat[c(3,140),]
# make pca
pca<-dudi.pca(PROC.parents,scannf=T,scale=F)
pourcent<-(pca$eig/sum(pca$eig))*100
# project other specimen on this pca
projection_f2<-suprow(pca,PROC.mat)
# extract pc coordinates for each individual
coord_parentspca<-cbind(rownames(projection_f2$lisup),projection_f2$lisup)

#####################################################################
## c. perform PCA on the genus
#####################################################################

# make procrustes superimposition
semi.landmarks<-as.matrix(read.table(file.choose(),h=T))
PROC<-gpagen(tot,curves=semi.landmarks)

# measure mean shape by individual
mean.shape.all<-array(NA,dim=c(32,2,length(unique(dimnames(PROC$coord)[[3]]))))
for (point in (1:32)){
  for (xy in (1:2)){
    res<-tapply(PROC$coord[point,xy,],dimnames(PROC$coord)[[3]],mean)
    mean.shape.all[point,xy,]<-res
  }
}
dimnames(mean.shape.all)<-list(NULL,NULL,names(res))

# perform PCA on the genus and project the F2 individuals on it
# create a matrix
PROC.mat<-two.d.array(mean.shape.all)
rownames(PROC.mat)<-dimnames(mean.shape.all)[[3]]
# extract data for wild specimen
wild.sp<-PROC.mat[c(5:45,181:220),]
# make PCA
pca_especes<-dudi.pca(wild.sp,scannf=T,scale=F)
pourcent<-(pca_especes$eig/sum(pca_especes$eig))*100
# project other specimen on this pca
projection_f2<-suprow(pca_especes,PROC.mat[c(1:4,46:180,221),])
projection_parents<-suprow(pca_especes,PROC.mat[c(3,221),])

# extract pc coordinates for each individual
coord_genuspca<-cbind(rownames(projection_f2$li),projection_f2$li)

#####################################################################
## d. extract univariate traits
#####################################################################

# extract landmarks coordinates
Xa<-hybrids[1,1,]
Xa<-tapply(hybrids[1,1,],dimnames(hybrids)[[3]],mean)
Ya<-hybrids[1,2,]
Ya<-tapply(hybrids[1,2,],dimnames(hybrids)[[3]],mean)
Xb<-hybrids[2,1,]
Xb<-tapply(hybrids[2,1,],dimnames(hybrids)[[3]],mean)
Yb<-hybrids[2,2,]
Yb<-tapply(hybrids[2,2,],dimnames(hybrids)[[3]],mean)

Xc<-hybrids[3,1,]
Xc<-tapply(hybrids[3,1,],dimnames(hybrids)[[3]],mean)
Yc<-hybrids[3,2,]
Yc<-tapply(hybrids[3,2,],dimnames(hybrids)[[3]],mean)
Xd<-hybrids[18,1,]
Xd<-tapply(hybrids[18,1,],dimnames(hybrids)[[3]],mean)
Yd<-hybrids[18,2,]
Yd<-tapply(hybrids[18,2,],dimnames(hybrids)[[3]],mean)

Xe<-hybrids[17,1,]
Xe<-tapply(hybrids[17,1,],dimnames(hybrids)[[3]],mean)
Ye<-hybrids[17,2,]
Ye<-tapply(hybrids[17,2,],dimnames(hybrids)[[3]],mean)
Xf<-hybrids[32,1,]
Xf<-tapply(hybrids[32,1,],dimnames(hybrids)[[3]],mean)
Yf<-hybrids[32,2,]
Yf<-tapply(hybrids[32,2,],dimnames(hybrids)[[3]],mean)

# calculate corolla tube openning
corolla.tube.openning<-sqrt(((Xc-Xd)*(Xc-Xd))+((Yc-Yd)*(Yc-Yd)))

# calculate corolla curvature
Xbbis<-Xb-(Xb-Xf)
Ybbis<-Yb-(Yb-Yf)
Xabis<-Xa-(Xb-Xf)
Yabis<-Ya-(Yb-Yf)
difAbisE<-Xabis-Xe 
dist_AbisBbis<-sqrt((Xabis-Xbbis)^2 + (Yabis -Ybbis)^2)
dist_AbisE<-sqrt((Xabis-Xe)^2 + (Yabis-Ye)^2)
dist_BbisE<-sqrt((Xbbis -Xe)^2 + (Ybbis -Ye)^2)
angle<-acos((dist_BbisE^2+dist_AbisBbis^2-dist_AbisE^2)/(2*dist_BbisE*dist_AbisBbis))*180/pi


#####################################################################
#####################################################################
# 2. measure between-traits correlations
#####################################################################
#####################################################################

data <- read.table(file="traits.txt",header=T)
colnames(data)
data<-data[,c(1:3,6:9,11,13,15:16)]
data_F2<-data[data[,2]=="F2",]
data_p<-data[data[,2]%in%c("P_auriculatum","selfed_F1","P_rupincola"),]

require(ggplot2)

# ggally ------------------------------------------------------------------

require(GGally)

# Threshold for Sidak correction of multiple tests
ntests = (ncol(data)-2)*(ncol(data)-3)/2
p_threshold = 1-(1-0.05)^(1/ntests)

number_col = ncol(data)
custom_plot <- ggpairs(data[,colnames(data)[3:number_col]],
                       upper="blank",lower="blank")
#upper and lower diagonal
for(i in 1:(number_col-3)) {
  for (j in 1:i) {
    if (i==j) next
    #lower diagonal
    atest <- cor.test(data[,(j+2)],data[,(i+2)])
    aplot <- ggplot2::ggplot(data, ggplot2::aes_string(x=colnames(data)[j+2], y=colnames(data)[i+2]))
    if(atest$p.value<p_threshold) {
      aplot <- aplot +
        ggplot2::geom_point(size = 1) +
        ggplot2::stat_smooth(method = "lm")
    }
    else {
      aplot <- aplot +
        ggplot2::geom_point(size = 1)
    }
    custom_plot <- putPlot(custom_plot, aplot, i, j)
    #upper diagonal
    temp <- data[,c((i+2),(j+2))]
    temp <- na.omit(temp)
    cor_obj <- cor.test(temp[,1],temp[,2])
    #    cor_signif <- symnum(cor_obj$p.value, corr = FALSE,
    #                         cutpoints = c(0, .001, .01, .05, 1),
    #                         symbols = c("***", "**", "*", ""))
    #    some_text <- paste(signif(cor_obj$estimate, 3),
    #                   cor_signif, sep="",collapse="")
    some_text <- signif(cor_obj$estimate, 3)                 
    if(cor_obj$p.value<p_threshold) {
      text_plot <- ggally_text(label = some_text, xP=0.5, yP=0.5,
                               mapping = ggplot2::aes_string(color="grey40",size=3,fontface=2),size=5) +
        theme(legend.position = "none", panel.grid.major = element_blank(), 
              axis.ticks = element_blank())      
    }
    else {
      text_plot <- ggally_text(label = some_text, xP=0.5, yP=0.5,
                               mapping = ggplot2::aes_string(color="grey40",size=3,fontface=1)) +
        theme(legend.position = "none", panel.grid.major = element_blank(), 
              axis.ticks = element_blank())
    }
    custom_plot <- putPlot(custom_plot, text_plot, j, i)
  }
}

#diagonal elements
for(i in 1:(number_col-3)) {
  aplot <- ggplot2::ggplot(data, ggplot2::aes_string(x=colnames(data)[i+2]))
  aplot <- aplot +
    ggplot2::geom_histogram(aes(y=..density..),fill="gray50") +
    ggplot2::geom_density(fill=NA,col="black",size=0.2) +
    ggplot2::geom_vline(xintercept=mean(data[data[,2]=="P_rupincola",(i+2)]),col="#d73027",size=0.7) +
    ggplot2::geom_vline(xintercept=mean(data[data[,2]=="P_auriculatum",(i+2)]),col="#1a9850",size=0.7) +
    ggplot2::geom_vline(xintercept=mean(data[data[,2]=="selfed_F1",(i+2)]),col="#984ea3",size=0.7)+
    ggplot2::theme(plot.title = element_blank())
  custom_plot <- putPlot(custom_plot, aplot, i, i)
}

#color
colPalette <- c("#ffeda0","#f03b20")
aplot <- ggplot2::ggplot(data[!is.na(data[,number_col]),], ggplot2::aes_string(x=colnames(data)[number_col]))
aplot <- aplot + geom_bar(fill=c("#ffeda0","#f03b20"),colour="black",size=0.3) +
  ggplot2::ggtitle(colnames(data)[number_col]) +
  ggplot2::theme(plot.title = element_blank())
custom_plot <- putPlot(custom_plot, aplot, (number_col-2), (number_col-2))

for(i in 1:(number_col-3)) {
  aplot <- ggplot2::ggplot(data[!is.na(data$couleur),], ggplot2::aes_string(x="couleur",y=colnames(data)[i+2]))
  aplot <- aplot + ggplot2::geom_boxplot(ggplot2::aes(fill=couleur),size=0.3,outlier.size = 0.8) +
    ggplot2::theme(legend.position="none") +
    ggplot2::scale_fill_manual(values=colPalette)
  custom_plot <- putPlot(custom_plot, aplot, (number_col-2), i)
  temp <- data[,c((i+2),number_col)]
  temp <- na.omit(temp)
  atest <- anova(lm(temp[,1]~temp[,2]))
  #  signif <- symnum(atest$Pr[1], corr = FALSE,
  #                       cutpoints = c(0, .001, .01, .05, 1),
  #                       symbols = c("p<=0.001", "p<=0.01", "p<=0.05", "p>0.05"))
  if(atest$Pr[1]<p_threshold) {
    text_plot <- ggally_text(label = "p <= 0.05", xP=0.5, yP=0.5,
                             mapping = ggplot2::aes_string(color="grey40",size=3,fontface=2)) +
      theme(legend.position = "none", panel.grid.major = element_blank(), 
            axis.ticks = element_blank())
  }
  else {
    text_plot <- ggally_text(label = "p > 0.05", xP=0.5, yP=0.5,
                             mapping = ggplot2::aes_string(color="grey40",size=3,fontface=1)) +
      theme(legend.position = "none", panel.grid.major = element_blank(), 
            axis.ticks = element_blank())  
  }
  custom_plot <- putPlot(custom_plot, text_plot, i, (number_col-2))
}

pdf("plot2.pdf",width=10,height=10)
custom_plot
dev.off()





